#!/usr/bin/env python3
"""
Simulador (MOCK) de respuesta SII para desarrollo sin AEAT (sin certificado).
- Lee un XML de envío (SOAP) y extrae las facturas (NumSerieFacturaEmisor, FechaExpedicionFacturaEmisor).
- Genera una respuesta SOAP "tipo AEAT" con una RespuestaLinea por factura.

Uso:
  python simulate_sii_response.py --request envio.xml --out respuesta_mock.xml
"""
from __future__ import annotations
import argparse
from datetime import datetime
from pathlib import Path
from lxml import etree

NS = {
    "soapenv": "http://schemas.xmlsoap.org/soap/envelope/",
    "sii": "https://www2.agenciatributaria.gob.es/static_files/common/internet/dep/aplicaciones/es/aeat/ssii/fact/ws/SuministroInformacion.xsd",
    "siiLR": "https://www2.agenciatributaria.gob.es/static_files/common/internet/dep/aplicaciones/es/aeat/ssii/fact/ws/SuministroLR.xsd",
    "siiR": "https://www2.agenciatributaria.gob.es/static_files/common/internet/dep/aplicaciones/es/aeat/ssii/fact/ws/RespuestaSuministro.xsd",
}

def q(prefix: str, tag: str) -> str:
    return f"{{{NS[prefix]}}}{tag}"

def extract_invoices(request_xml: Path):
    doc = etree.parse(str(request_xml))
    nodes = doc.xpath("//*[local-name()='RegistroLRFacturasEmitidas']")
    out = []
    for n in nodes:
        num = n.xpath(".//*[local-name()='NumSerieFacturaEmisor']/text()")
        fec = n.xpath(".//*[local-name()='FechaExpedicionFacturaEmisor']/text()")
        out.append({
            "NumSerieFacturaEmisor": (num[0] if num else "SIN_NUMERO"),
            "FechaExpedicionFacturaEmisor": (fec[0] if fec else ""),
        })
    return out

def build_response(invoices, estado_envio: str, csv: str):
    env = etree.Element(q("soapenv", "Envelope"), nsmap=NS)
    etree.SubElement(env, q("soapenv", "Header"))
    body = etree.SubElement(env, q("soapenv", "Body"))

    resp = etree.SubElement(body, q("siiR", "RespuestaSuministroLRFacturasEmitidas"))

    cab = etree.SubElement(resp, q("sii", "Cabecera"))
    etree.SubElement(cab, q("sii", "IDVersionSii")).text = "1.1"
    etree.SubElement(cab, q("sii", "TipoComunicacion")).text = "A0"

    etree.SubElement(resp, q("siiR", "EstadoEnvio")).text = estado_envio
    etree.SubElement(resp, q("siiR", "CSV")).text = csv
    etree.SubElement(resp, q("siiR", "TimestampPresentacion")).text = datetime.now().strftime("%Y-%m-%dT%H:%M:%S")

    for inv in invoices:
        linea = etree.SubElement(resp, q("siiR", "RespuestaLinea"))
        idf = etree.SubElement(linea, q("siiLR", "IDFactura"))
        etree.SubElement(idf, q("sii", "NumSerieFacturaEmisor")).text = inv["NumSerieFacturaEmisor"]
        etree.SubElement(idf, q("sii", "FechaExpedicionFacturaEmisor")).text = inv["FechaExpedicionFacturaEmisor"]

        etree.SubElement(linea, q("siiR", "EstadoRegistro")).text = "Correcto"
        etree.SubElement(linea, q("siiR", "CodigoErrorRegistro")).text = ""
        etree.SubElement(linea, q("siiR", "DescripcionErrorRegistro")).text = ""

    return etree.tostring(env, xml_declaration=True, encoding="utf-8", pretty_print=True)

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--request", required=True, help="XML de envío (SOAP)")
    ap.add_argument("--out", default="respuesta_mock.xml", help="Ruta de salida")
    ap.add_argument("--estado", default="Correcto", choices=["Correcto","AceptadoConErrores","Incorrecto"])
    ap.add_argument("--csv", default="CSV_DEMO_0000000000")
    args = ap.parse_args()

    req = Path(args.request)
    if not req.exists():
        raise SystemExit(f"No existe: {req}")

    invoices = extract_invoices(req)
    data = build_response(invoices, args.estado, args.csv)
    Path(args.out).write_bytes(data)
    print(f"OK -> {args.out} (líneas: {len(invoices)})")

if __name__ == "__main__":
    main()
