#!/usr/bin/env python3
"""
excel_to_sii.py
Convierte una plantilla Excel (openpyxl) a un XML SOAP SII (Facturas Emitidas).

ANTES este script tenía NIF/NOMBRE y rutas hardcodeadas.
Ahora admite argumentos para poder usarlo en modo multiempresa/multicuenta.

Ejemplo:
  python excel_to_sii.py --xlsx uploads/in.xlsx --out outputs/request.xml --titular-nif A12345678 --titular-nombre "Mi Empresa SL"
"""
from __future__ import annotations

import argparse
from datetime import date, datetime
from pathlib import Path
from xml.etree.ElementTree import Element, SubElement, tostring, register_namespace

from openpyxl import load_workbook


def s(x) -> str:
    """Convierte a string limpio; devuelve "" si viene None."""
    if x is None:
        return ""
    return str(x).strip()


def is_empty(x) -> bool:
    return s(x) == ""


def to_float(x, default: float = 0.0) -> float:
    try:
        if x is None or x == "":
            return default
        return float(x)
    except Exception:
        return default


def fmt_date_es(d) -> str:
    if isinstance(d, datetime):
        return d.strftime("%d-%m-%Y")
    if isinstance(d, date):
        return d.strftime("%d-%m-%Y")
    return str(d)


def fmt_periodo(p) -> str:
    ss = str(p).strip()
    if ss.isdigit():
        return str(int(ss)).zfill(2)
    return ss.zfill(2)


NS = {
    "soapenv": "http://schemas.xmlsoap.org/soap/envelope/",
    "siiLR": "https://www2.agenciatributaria.gob.es/static_files/common/internet/dep/aplicaciones/es/aeat/ssii/fact/ws/SuministroLR.xsd",
    "sii": "https://www2.agenciatributaria.gob.es/static_files/common/internet/dep/aplicaciones/es/aeat/ssii/fact/ws/SuministroInformacion.xsd",
}
for p, uri in NS.items():
    register_namespace(p, uri)


def q(prefix: str, tag: str) -> str:
    return f"{{{NS[prefix]}}}{tag}"


def build_xml(xlsx_file: Path, titular_nif: str, titular_nombre: str) -> bytes:
    # Hojas
    SHEET_FACT = "FacturasEmitidas"
    SHEET_IVA = "IVA_Detalle"

    wb = load_workbook(str(xlsx_file), data_only=True)
    fe = wb[SHEET_FACT]
    iva = wb[SHEET_IVA]

    # IVA por FacturaID
    iva_by_id: dict[str, list[dict]] = {}
    for row in iva.iter_rows(min_row=2, values_only=True):
        if not row or not row[0]:
            continue
        factura_id = str(row[0]).strip()
        tipo_no_exenta = str(row[1] or "S1")
        base = to_float(row[2])
        tipo = to_float(row[3])
        cuota = to_float(row[4])
        iva_by_id.setdefault(factura_id, []).append(
            {"TipoNoExenta": tipo_no_exenta, "Base": base, "Tipo": tipo, "Cuota": cuota}
        )

    # SOAP + payload
    env = Element(q("soapenv", "Envelope"))
    SubElement(env, q("soapenv", "Header"))
    body = SubElement(env, q("soapenv", "Body"))
    root = SubElement(body, q("siiLR", "SuministroLRFacturasEmitidas"))

    # Cabecera
    cab = SubElement(root, q("sii", "Cabecera"))
    SubElement(cab, q("sii", "IDVersionSii")).text = "1.1"
    tit = SubElement(cab, q("sii", "Titular"))
    SubElement(tit, q("sii", "NombreRazon")).text = titular_nombre
    # NIF es opcional en algunos escenarios, pero el XSD de cabecera suele requerir identificación.
    # Si tu caso permite "IDOtro" o NIF vacío, esto se ajusta en una iteración posterior.
    SubElement(tit, q("sii", "NIF")).text = titular_nif
    SubElement(cab, q("sii", "TipoComunicacion")).text = "A0"

    # Registros: FacturasEmitidas
    for r in fe.iter_rows(min_row=2, values_only=True):
        # Columnas esperadas (según tu plantilla). Si hay cambios, ajustamos el mapeo.
        # 0:FacturaID 1:Ejercicio 2:Periodo 3:NumSerie 4:FechaExped 5:FechaOp 6:Nombre 7:NIF
        # 8:TipoFactura 9:ClaveRegimen 10:ImporteTotal ...
        if not r or is_empty(r[0]):
            continue

        factura_id = s(r[0])
        ejercicio = s(r[1])
        periodo = fmt_periodo(r[2])
        num_serie = s(r[3])
        fecha_exped = fmt_date_es(r[4])
        fecha_op = fmt_date_es(r[5]) if not is_empty(r[5]) else fecha_exped
        contrap_nombre = s(r[6])
        contrap_nif = s(r[7])
        tipo_factura = s(r[8]) or "F1"
        clave_regimen = s(r[9]) or "01"
        importe_total = f"{to_float(r[10]):.2f}" if len(r) > 10 else "0.00"

        reg = SubElement(root, q("siiLR", "RegistroLRFacturasEmitidas"))
        per = SubElement(reg, q("sii", "PeriodoImpositivo"))
        SubElement(per, q("sii", "Ejercicio")).text = ejercicio
        SubElement(per, q("sii", "Periodo")).text = periodo

        idfact = SubElement(reg, q("siiLR", "IDFactura"))
        em = SubElement(idfact, q("sii", "IDEmisorFactura"))
        # En emitidas, el emisor suele ser el propio titular; dejamos en blanco para que no choque con tu lógica.
        # SubElement(em, q("sii", "NIF")).text = titular_nif

        numserie = SubElement(idfact, q("sii", "NumSerieFacturaEmisor"))
        numserie.text = num_serie
        SubElement(idfact, q("sii", "FechaExpedicionFacturaEmisor")).text = fecha_exped

        fact = SubElement(reg, q("siiLR", "FacturaExpedida"))
        SubElement(fact, q("sii", "TipoFactura")).text = tipo_factura
        SubElement(fact, q("sii", "ClaveRegimenEspecialOTrascendencia")).text = clave_regimen
        SubElement(fact, q("sii", "ImporteTotal")).text = importe_total

        # Contraparte: si no hay NIF, se puede usar IDOtro (R5 / extranjero). Aquí dejamos el NIF si existe.
        if contrap_nombre or contrap_nif:
            contr = SubElement(fact, q("sii", "Contraparte"))
            if contrap_nombre:
                SubElement(contr, q("sii", "NombreRazon")).text = contrap_nombre
            if contrap_nif:
                SubElement(contr, q("sii", "NIF")).text = contrap_nif

        # Desglose IVA (si existe)
        desg = SubElement(fact, q("sii", "TipoDesglose"))
        desg2 = SubElement(desg, q("sii", "DesgloseIVA"))
        dets = iva_by_id.get(factura_id, [])
        for d in dets:
            det = SubElement(desg2, q("sii", "DetalleIVA"))
            SubElement(det, q("sii", "TipoImpositivo")).text = f"{d['Tipo']:.2f}"
            SubElement(det, q("sii", "BaseImponible")).text = f"{d['Base']:.2f}"
            SubElement(det, q("sii", "CuotaRepercutida")).text = f"{d['Cuota']:.2f}"

        # Fecha operación (si procede)
        SubElement(fact, q("sii", "FechaOperacion")).text = fecha_op

    return tostring(env, encoding="utf-8", xml_declaration=True)


def main() -> None:
    ap = argparse.ArgumentParser()
    ap.add_argument("--xlsx", required=True, help="Ruta del Excel de entrada")
    ap.add_argument("--out", required=True, help="Ruta del XML SOAP de salida")
    ap.add_argument("--titular-nif", required=True, help="NIF del titular (empresa)")
    ap.add_argument("--titular-nombre", required=True, help="Nombre/Razón Social del titular")
    args = ap.parse_args()

    xlsx = Path(args.xlsx)
    if not xlsx.exists():
        raise SystemExit(f"ERROR: No existe el Excel: {xlsx}")

    out = Path(args.out)
    out.parent.mkdir(parents=True, exist_ok=True)

    xml_bytes = build_xml(xlsx, args.titular_nif.strip(), args.titular_nombre.strip())
    out.write_bytes(xml_bytes)
    print(f"OK -> {out}")


if __name__ == "__main__":
    main()
