<?php
declare(strict_types=1);
require __DIR__ . '/../app/bootstrap.php';
require_login();
csrf_check();

$pdo = db();
$tenantId = (int)$_SESSION['user']['tenant_id'];

$schema = $_POST['schema'] ?? 'SuministroLR.xsd';
if (!in_array($schema, ['SuministroLR.xsd','SuministroInformacion.xsd'], true)) $schema = 'SuministroLR.xsd';

if (empty($_FILES['xlsx']) || $_FILES['xlsx']['error'] !== UPLOAD_ERR_OK) {
  http_response_code(400);
  echo "Error subiendo el archivo.";
  exit;
}

$tenantSt = $pdo->prepare("SELECT * FROM tenants WHERE id=?");
$tenantSt->execute([$tenantId]);
$tenant = $tenantSt->fetch();
if (!$tenant) { http_response_code(400); echo "Tenant inválido."; exit; }

// Crear job
$jobId = bin2hex(random_bytes(8));
$pdo->prepare("INSERT INTO sii_jobs (id, tenant_id, user_id, status, schema_name, created_at) VALUES (?,?,?,?,?,NOW())")
    ->execute([$jobId, $tenantId, (int)$_SESSION['user']['id'], 'running', $schema]);

// Directorios
$td = tenant_dir($tenantId);
ensure_dir($td);
$uploads = $td . DIRECTORY_SEPARATOR . 'uploads';
$outputs = $td . DIRECTORY_SEPARATOR . 'outputs';
$logs    = $td . DIRECTORY_SEPARATOR . 'logs';
ensure_dir($uploads); ensure_dir($outputs); ensure_dir($logs);

$xlsxPath = $uploads . DIRECTORY_SEPARATOR . $jobId . '.xlsx';
if (!move_uploaded_file($_FILES['xlsx']['tmp_name'], $xlsxPath)) {
  $pdo->prepare("UPDATE sii_jobs SET status='error', finished_at=NOW(), error_message=? WHERE id=? AND tenant_id=?")
      ->execute(['No pude mover el archivo subido.', $jobId, $tenantId]);
  echo "No pude guardar el archivo.";
  exit;
}

$requestXml = $outputs . DIRECTORY_SEPARATOR . $jobId . '_request.xml';
$payloadXml = $outputs . DIRECTORY_SEPARATOR . $jobId . '_payload.xml';
$logFile    = $logs . DIRECTORY_SEPARATOR . $jobId . '.log';

$python = trim((string)config('python_bin'));
if ($python === '') { $python = 'python'; }
$engine = rtrim(config('engine_path'), '/\\');
$excelPy = $engine . DIRECTORY_SEPARATOR . 'excel_to_sii.py';
$extractPy = $engine . DIRECTORY_SEPARATOR . 'validador' . DIRECTORY_SEPARATOR . 'extract_sii_payload.py';
$validatePy = $engine . DIRECTORY_SEPARATOR . 'validador' . DIRECTORY_SEPARATOR . 'validate_sii_xsd.py';
$cacheDir = $engine . DIRECTORY_SEPARATOR . 'validador' . DIRECTORY_SEPARATOR . 'xsd_cache';

// ---------------------------------------------------------
// Windows: evita errores de encoding (cp1252) al imprimir unicode/emoji
// y fuerza salida UTF-8 en scripts Python.
// ---------------------------------------------------------
putenv('PYTHONUTF8=1');
putenv('PYTHONIOENCODING=utf-8');

// "<python.exe>" -X utf8
$pythonCmd = '"' . str_replace('"', '', $python) . '" -X utf8';

function run_cmd(string $cmd, string $logFile): int {
  $out = [];
  $code = 0;
  exec($cmd . " 2>&1", $out, $code);
  file_put_contents($logFile, $cmd . PHP_EOL . implode(PHP_EOL, $out) . PHP_EOL . "EXIT=" . $code . PHP_EOL, FILE_APPEND);
  return $code;
}

/**
 * Parche compatibilidad XSD:
 * Algunos generadores usan <PeriodoImpositivo> pero el XSD de SuministroLR espera <PeriodoLiquidacion>.
 * Este fix renombra el nodo manteniendo namespace y contenido.
 */
function fix_periodo_liquidacion(string $xmlPath, string $logFile): bool {
  if (!is_file($xmlPath)) {
    file_put_contents($logFile, "FIX: no existe XML: $xmlPath" . PHP_EOL, FILE_APPEND);
    return false;
  }

  $dom = new DOMDocument();
  $dom->preserveWhiteSpace = false;
  $dom->formatOutput = true;

  libxml_use_internal_errors(true);
  $ok = $dom->load($xmlPath);
  if (!$ok) {
    $errs = libxml_get_errors();
    libxml_clear_errors();
    foreach ($errs as $e) {
      file_put_contents($logFile, "FIX XML parse error: " . trim($e->message) . PHP_EOL, FILE_APPEND);
    }
    return false;
  }

  $xp = new DOMXPath($dom);
  $nodes = $xp->query('//*[local-name()="PeriodoImpositivo"]');

  $changed = 0;
  if ($nodes) {
    // iterar de forma segura (NodeList se "rompe" si modificas en caliente)
    $toChange = [];
    foreach ($nodes as $n) { $toChange[] = $n; }

    foreach ($toChange as $node) {
      if (!$node instanceof DOMElement) continue;
      $ns = $node->namespaceURI;
      $new = $dom->createElementNS($ns ?: null, 'PeriodoLiquidacion');

      // copiar atributos (si los hubiera)
      if ($node->hasAttributes()) {
        foreach ($node->attributes as $attr) {
          if (!$attr instanceof DOMAttr) continue;
          $new->setAttributeNS($attr->namespaceURI ?: null, $attr->nodeName, $attr->nodeValue);
        }
      }

      // mover hijos
      while ($node->firstChild) {
        $new->appendChild($node->firstChild);
      }

      $node->parentNode?->replaceChild($new, $node);
      $changed++;
    }
  }

  if ($changed > 0) {
    $dom->save($xmlPath);
  }

  file_put_contents($logFile, "FIX PeriodoImpositivo->PeriodoLiquidacion: $changed" . PHP_EOL, FILE_APPEND);
  return true;
}

file_put_contents($logFile, "JOB $jobId" . PHP_EOL);

$cmd1 = $pythonCmd . " " . escapeshellarg($excelPy)
  . " --xlsx " . escapeshellarg($xlsxPath)
  . " --out " . escapeshellarg($requestXml)
  . " --titular-nif " . escapeshellarg($tenant['nif'])
  . " --titular-nombre " . escapeshellarg($tenant['name']);
$c1 = run_cmd($cmd1, $logFile);

$cmd2 = $pythonCmd . " " . escapeshellarg($extractPy)
  . " --in " . escapeshellarg($requestXml)
  . " --out " . escapeshellarg($payloadXml)
  . " --tag " . escapeshellarg('SuministroLRFacturasEmitidas');
$c2 = ($c1===0) ? run_cmd($cmd2, $logFile) : 1;
if ($c2 === 0) {
  // parche XSD (PeriodoImpositivo -> PeriodoLiquidacion)
  if (!fix_periodo_liquidacion($payloadXml, $logFile)) {
    $c2 = 1;
  }
}

$cmd3 = $pythonCmd . " " . escapeshellarg($validatePy)
  . " --xml " . escapeshellarg($payloadXml)
  . " --schema " . escapeshellarg($schema)
  . " --cache " . escapeshellarg($cacheDir);
$c3 = ($c2===0) ? run_cmd($cmd3, $logFile) : 1;

$status = ($c1===0 && $c2===0 && $c3===0) ? 'ok' : 'error';
$errMsg = $status==='ok' ? null : 'Fallo en conversión/validación. Revisa el log.';
$pdo->prepare("UPDATE sii_jobs SET status=?, finished_at=NOW(), request_xml_path=?, payload_xml_path=?, log_path=?, error_message=? WHERE id=? AND tenant_id=?")
    ->execute([$status, $requestXml, $payloadXml, $logFile, $errMsg, $jobId, $tenantId]);

header('Location: ' . base_url() . '/job.php?id=' . urlencode($jobId));
