<?php
/**
 * Bootstrap principal - estable para includes múltiples y XAMPP.
 * Compatible con código legacy del SII.
 *
 * Incluye:
 * - session cookie path "/"
 * - config(), db(), url(), redirect(), e()
 * - CSRF (csrf_verify + alias csrf_check)
 * - auth helpers (require_login, require_admin)
 * - ensure_dir() + tenant_dir()  ✅ (para run_job.php)
 */

if (defined('APP_BOOTSTRAPPED')) { return; }
define('APP_BOOTSTRAPPED', true);

/* =========================================================
 * SESIÓN
 * =======================================================*/
if (session_status() === PHP_SESSION_NONE) {
    if (!headers_sent()) {
        session_set_cookie_params([
            'lifetime' => 0,
            'path' => '/',        // CLAVE para no perder sesión entre /public y /
            'httponly' => true,
            'samesite' => 'Lax',
        ]);
    }
    session_start();
}

/* =========================================================
 * CONFIG
 * =======================================================*/
if (!function_exists('config')) {
    function config(string $key = null) {
        static $config = null;

        if ($config === null) {
            $file = __DIR__ . '/config.php';
            if (!file_exists($file)) {
                throw new Exception('app/config.php no encontrado');
            }
            $config = require $file;
        }

        if ($key === null) return $config;

        $value = $config;
        foreach (explode('.', $key) as $p) {
            if (!is_array($value) || !array_key_exists($p, $value)) return null;
            $value = $value[$p];
        }
        return $value;
    }
}

/* =========================================================
 * URLS
 * =======================================================*/
if (!function_exists('base_url')) {
    function base_url(): string {
        $cfg = config('app.base_url');
        if (is_string($cfg) && $cfg !== '') return rtrim($cfg, '/');

        $https = !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off';
        $scheme = $https ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';

        $path = rtrim(str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '')), '/');
        return $scheme . '://' . $host . $path;
    }
}

if (!function_exists('url')) {
    function url(string $p = ''): string {
        $p = ltrim($p, '/');
        return $p === '' ? rtrim(base_url(), '/') : rtrim(base_url(), '/') . '/' . $p;
    }
}

/* =========================================================
 * DATABASE
 * =======================================================*/
if (!function_exists('db')) {
    function db(): PDO {
        static $pdo = null;

        if (!$pdo) {
            $c = config('db');
            if (!$c) throw new Exception('Config db no encontrada');

            $pdo = new PDO(
                "mysql:host={$c['host']};dbname={$c['name']};charset=utf8mb4",
                $c['user'],
                $c['pass'],
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false,
                ]
            );
        }

        return $pdo;
    }
}

/* =========================================================
 * UTILIDADES
 * =======================================================*/
if (!function_exists('redirect')) {
    function redirect(string $to): void {
        $isAbs = preg_match('#^https?://#i', $to);
        header('Location: ' . ($isAbs ? $to : url($to)));
        exit;
    }
}

if (!function_exists('e')) {
    function e(string $v): string {
        return htmlspecialchars($v, ENT_QUOTES, 'UTF-8');
    }
}

/**
 * ✅ ensure_dir()
 * - Crea el directorio si no existe
 * - Devuelve la ruta (realpath si es posible)
 */
if (!function_exists('ensure_dir')) {
    function ensure_dir(string $dir, int $mode = 0775): string {
        if ($dir === '') throw new InvalidArgumentException('ensure_dir(): dir vacío');

        if (!is_dir($dir)) {
            if (!@mkdir($dir, $mode, true) && !is_dir($dir)) {
                throw new RuntimeException('No se pudo crear el directorio: ' . $dir);
            }
        }

        $rp = realpath($dir);
        return $rp !== false ? $rp : $dir;
    }
}

/* =========================================================
 * AUTH / ROLES
 * =======================================================*/
if (!function_exists('auth')) {
    function auth(): bool {
        return !empty($_SESSION['user_id']);
    }
}

if (!function_exists('current_user_id')) {
    function current_user_id(): ?int {
        return isset($_SESSION['user_id']) ? (int)$_SESSION['user_id'] : null;
    }
}

if (!function_exists('current_tenant_id')) {
    function current_tenant_id(): ?int {
        return isset($_SESSION['tenant_id']) ? (int)$_SESSION['tenant_id'] : null;
    }
}

if (!function_exists('current_role')) {
    function current_role(): string {
        return (string)($_SESSION['role'] ?? 'user');
    }
}

if (!function_exists('require_auth')) {
    function require_auth(): void {
        if (!auth()) redirect('login.php');
    }
}

if (!function_exists('require_login')) {
    function require_login(): void { require_auth(); }
}

if (!function_exists('require_admin')) {
    function require_admin(): void {
        require_auth();
        if (current_role() !== 'admin') {
            http_response_code(403);
            exit('Acceso denegado');
        }
    }
}

/* =========================================================
 * CSRF
 * =======================================================*/
if (!function_exists('csrf_token')) {
    function csrf_token(): string {
        if (empty($_SESSION['_csrf'])) {
            $_SESSION['_csrf'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['_csrf'];
    }
}

if (!function_exists('csrf_field')) {
    function csrf_field(): string {
        return '<input type="hidden" name="_csrf" value="' . e(csrf_token()) . '">';
    }
}

if (!function_exists('csrf_verify')) {
    function csrf_verify(): void {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') return;

        $sent = $_POST['_csrf'] ?? $_POST['csrf'] ?? '';
        $ok = isset($_SESSION['_csrf']) && is_string($sent) && $sent !== '' && hash_equals($_SESSION['_csrf'], $sent);

        if (!$ok) {
            http_response_code(403);
            exit('CSRF inválido');
        }
    }
}

if (!function_exists('csrf_check')) {
    function csrf_check(): void { csrf_verify(); }
}

/* =========================================================
 * TENANT DIR
 * =======================================================*/
if (!function_exists('tenant_dir')) {
    function tenant_dir(?int $tenantId = null): string {
        $tenantId ??= current_tenant_id();
        if (!$tenantId) {
            throw new RuntimeException('Tenant no definido');
        }

        $base = dirname(__DIR__) . '/data/tenants';
        $dir  = $base . '/' . $tenantId;

        // Estructura típica
        ensure_dir($dir);
        ensure_dir($dir . '/inputs');
        ensure_dir($dir . '/outputs');
        ensure_dir($dir . '/logs');
        ensure_dir($dir . '/tmp');

        $rp = realpath($dir);
        return $rp !== false ? $rp : $dir;
    }
}
